package nemosofts.notes.app.widget;

import android.app.PendingIntent;
import android.appwidget.AppWidgetManager;
import android.appwidget.AppWidgetProvider;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.view.View;
import android.widget.RemoteViews;

import nemosofts.notes.app.R;
import nemosofts.notes.app.activity.CreateNoteActivity;
import nemosofts.notes.app.activity.MainActivity;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.utils.helper.DBHelper;

public class NoteWidgetProvider extends AppWidgetProvider {

    private static final String PREFS_NAME = "note_widget_prefs";
    private static final String PREF_PREFIX_KEY = "widget_noteId_";

    @Override
    public void onUpdate(Context context, AppWidgetManager appWidgetManager, int[] appWidgetIds) {
        for (int appWidgetId : appWidgetIds) {
            updateAppWidget(context, appWidgetManager, appWidgetId);
        }
    }

    @Override
    public void onDeleted(Context context, int[] appWidgetIds) {
        for (int appWidgetId : appWidgetIds) {
            deleteNoteIdPref(context, appWidgetId);
        }
        super.onDeleted(context, appWidgetIds);
    }

    public static void updateAppWidget(Context context, AppWidgetManager appWidgetManager, int appWidgetId) {
        int noteId = loadNoteIdPref(context, appWidgetId);
        RemoteViews views = new RemoteViews(context.getPackageName(), R.layout.widget_note);

        Note noteToShow = null;
        DBHelper dbHelper = new DBHelper(context);
        if (noteId != -1) {
            noteToShow = dbHelper.getNoteById(noteId);
        }
        if (noteToShow == null) {
            java.util.List<Note> all = dbHelper.getAllNotes();
            if (all != null && !all.isEmpty()) {
                noteToShow = all.get(0);
                saveNoteIdPref(context, appWidgetId, noteToShow.getId());
            }
        }

        if (noteToShow == null) {
            views.setTextViewText(R.id.widget_title, context.getString(R.string.app_name));
            views.setTextViewText(R.id.widget_subtitle, context.getString(R.string.search_note));
            views.setTextViewText(R.id.widget_date, context.getString(R.string.add_categories));
            views.setViewVisibility(R.id.widget_badge_link, View.GONE);
            views.setViewVisibility(R.id.widget_badge_audio, View.GONE);
            views.setViewVisibility(R.id.widget_badge_reminder, View.GONE);
            setOpenIntent(context, views, appWidgetId, null);
        } else {
            Note note = noteToShow;
            views.setTextViewText(R.id.widget_title, safeText(note.getTitle(), context.getString(R.string.note_title)));
            views.setTextViewText(R.id.widget_subtitle, safeText(note.getSubtitle(), ""));
            views.setTextViewText(R.id.widget_date, safeText(note.getDateTime(), ""));

            views.setViewVisibility(R.id.widget_badge_link,
                    note.getWebLink() != null && !note.getWebLink().trim().isEmpty() ? View.VISIBLE : View.GONE);
            views.setViewVisibility(R.id.widget_badge_audio,
                    note.getAudioPath() != null && !note.getAudioPath().trim().isEmpty() ? View.VISIBLE : View.GONE);
            views.setViewVisibility(R.id.widget_badge_reminder,
                    note.isReminderEnabled() && note.getReminderTime() > 0 ? View.VISIBLE : View.GONE);

            setOpenIntent(context, views, appWidgetId, note);
        }

        setNewNoteIntent(context, views, appWidgetId);
        appWidgetManager.updateAppWidget(appWidgetId, views);
    }

    private static void setOpenIntent(Context context, RemoteViews views, int widgetId, Note note) {
        Intent intent;
        if (note != null) {
            intent = new Intent(context, CreateNoteActivity.class);
            intent.putExtra("isViewOrUpdate", true);
            intent.putExtra("note", note);
        } else {
            intent = new Intent(context, MainActivity.class);
        }
        PendingIntent pendingIntent = PendingIntent.getActivity(
                context,
                widgetId,
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE
        );
        views.setOnClickPendingIntent(R.id.widget_root, pendingIntent);
    }

    private static void setNewNoteIntent(Context context, RemoteViews views, int widgetId) {
        Intent intent = new Intent(context, CreateNoteActivity.class);
        PendingIntent pendingIntent = PendingIntent.getActivity(
                context,
                widgetId + 10000,
                intent,
                PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE
        );
        views.setOnClickPendingIntent(R.id.widget_add, pendingIntent);
    }

    private static String safeText(String value, String fallback) {
        if (value == null || value.trim().isEmpty()) {
            return fallback;
        }
        return value;
    }

    public static void saveNoteIdPref(Context context, int appWidgetId, int noteId) {
        SharedPreferences.Editor prefs = context.getSharedPreferences(PREFS_NAME, 0).edit();
        prefs.putInt(PREF_PREFIX_KEY + appWidgetId, noteId);
        prefs.apply();
    }

    public static int loadNoteIdPref(Context context, int appWidgetId) {
        SharedPreferences prefs = context.getSharedPreferences(PREFS_NAME, 0);
        return prefs.getInt(PREF_PREFIX_KEY + appWidgetId, -1);
    }

    public static void deleteNoteIdPref(Context context, int appWidgetId) {
        SharedPreferences.Editor prefs = context.getSharedPreferences(PREFS_NAME, 0).edit();
        prefs.remove(PREF_PREFIX_KEY + appWidgetId);
        prefs.apply();
    }

    public static void requestUpdate(Context context) {
        AppWidgetManager appWidgetManager = AppWidgetManager.getInstance(context);
        ComponentName thisWidget = new ComponentName(context, NoteWidgetProvider.class);
        int[] appWidgetIds = appWidgetManager.getAppWidgetIds(thisWidget);
        if (appWidgetIds != null) {
            for (int appWidgetId : appWidgetIds) {
                updateAppWidget(context, appWidgetManager, appWidgetId);
            }
        }
    }
}
